<?php
// stamp.php (Complete Stamp & Signature Management Tool with File Upload - FINAL, SESSION-BASED ID FIX)

// Start the session to access the Principal ID
session_start();

// ======================================================================
// 1. CONFIGURATION AND INITIAL SETUP
// ======================================================================

// --- Database Connection Details ---
define('DB_HOST', 'localhost'); 
define('DB_USER', 'thevkin_schlautdb'); 
define('DB_PASS', 'VK_AutSchlDb'); 
define('DB_NAME', 'thevkin_schlautdb');

// --- Logged-in Principal ID (The CORRECT Fix) ---
// We retrieve the Principal ID string (e.g., #VKPR01 or #VKPRNC01) from the session.
$logged_in_principal_id = $_SESSION['prnc_id'] ?? null; 

// --- CRITICAL CHECK: Ensure a Principal ID exists ---
if (empty($logged_in_principal_id)) {
    // If the ID is missing, terminate execution or redirect to login.
    die("Authentication Error: Principal ID is missing from the session.");
}

// --- Upload Directory Configuration ---
// IMPORTANT: Ensure this directory exists and is writable (e.g., chmod 775 or 777)
define('UPLOAD_DIR', 'uploads/stamps/'); 

// --- Variables for Form and Status ---
$school_name = $stamp_path = $signature_path = '';
$record_exists = false;
$status_message = '';
$status_class = '';
$error = '';

// Allowed image file types and size
$allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
$max_file_size = 524288; // 512KB

// ======================================================================
// 2. DATABASE CONNECTION & UTILITIES
// ======================================================================

function get_auth_db_connection() {
    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    if ($conn->connect_error) {
        error_log("Connection failed: " . $conn->connect_error);
        return false;
    }
    return $conn;
}

$conn = get_auth_db_connection();

if (!$conn) {
    // If connection fails, display error but proceed to HTML (to show the setup error)
    $error .= "Database connection error. Please check configuration.";
} else {
    // --- SQL Query to Create the Stamp Table (VARCHAR ID) ---
    // principal_id is VARCHAR(50) to store strings like #VKPR01
    $sql_create_table = "
    CREATE TABLE IF NOT EXISTS school_stamps (
        id INT AUTO_INCREMENT PRIMARY KEY,
        school_name VARCHAR(255) NOT NULL,
        stamp_image_path VARCHAR(255) NULL COMMENT 'Path to the school stamp image',
        signature_image_path VARCHAR(255) NULL COMMENT 'Path to the principal signature image',
        principal_id VARCHAR(50) NOT NULL UNIQUE COMMENT 'The prnc_id string e.g., #VKPR01 or #VKPRNC01',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    );
    ";
    if (!$conn->query($sql_create_table)) {
        $error .= "Could not confirm/create table structure: " . $conn->error . "<br>";
    }
}


// ======================================================================
// 3. FILE UPLOAD HANDLER FUNCTION
// ======================================================================

function handle_upload($file_key, $principal_id, $type, $allowed_types, $max_size) {
    global $UPLOAD_DIR; 

    if (empty($_FILES[$file_key]['name'])) {
        return ['success' => true, 'message' => "$type file not selected.", 'path' => null];
    }

    $file = $_FILES[$file_key];
    
    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['success' => false, 'message' => "$type upload failed with error code: " . $file['error']];
    }

    // Check file size
    if ($file['size'] > $max_size) {
        return ['success' => false, 'message' => "$type file size exceeds " . ($max_size / 1024) . "KB."];
    }
    
    // --- MIME Type Check (Finfo fallback for broader compatibility) ---
    $mime_type = null;
    if (function_exists('finfo_open')) {
        $file_info = finfo_open(FILEINFO_MIME_TYPE);
        $mime_type = finfo_file($file_info, $file['tmp_name']);
        finfo_close($file_info);
    } else {
        $mime_type = $file['type'];
        error_log("Warning: Using less secure MIME type check for $type. Enable Fileinfo extension for security.");
    }
    
    if ($mime_type === null || !in_array($mime_type, $allowed_types)) {
        return ['success' => false, 'message' => "$type file type '$mime_type' is not allowed."];
    }
    // --- End MIME Check ---
    
    // Generate unique file name using the STRING ID
    $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    // Sanitize the ID for use in a file path (removing '#' and special chars)
    $safe_principal_id = preg_replace('/[^a-zA-Z0-9_-]/', '_', $principal_id); 
    $target_filename = $safe_principal_id . '_' . $type . '.' . strtolower($ext);
    $target_path = UPLOAD_DIR . $target_filename;

    // Ensure upload directory exists
    if (!is_dir(UPLOAD_DIR)) {
        if (!mkdir(UPLOAD_DIR, 0775, true)) {
             return ['success' => false, 'message' => "Failed to create upload directory: " . UPLOAD_DIR];
        }
    }

    // Move the file
    if (move_uploaded_file($file['tmp_name'], $target_path)) {
        return ['success' => true, 'message' => "$type uploaded.", 'path' => $target_path];
    } else {
        return ['success' => false, 'message' => "Failed to move uploaded $type file. Check permissions on " . UPLOAD_DIR];
    }
}

// ======================================================================
// 4. HANDLE FORM SUBMISSION (ADD or EDIT/UPDATE)
// ======================================================================

// Only attempt DB operations if connection is successful
if ($conn && $_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['save_stamp_details'])) {
    
    // 4.1. Sanitize base input
    $new_school_name = trim($_POST['school_name']);
    
    if (empty($new_school_name)) {
        $status_message = "School Name is required.";
        $status_class = 'alert-error';
        goto skip_db_update;
    }

    // 4.2. Handle file uploads
    $upload_result_stamp = handle_upload('stamp_file', $logged_in_principal_id, 'stamp', $allowed_types, $max_file_size);
    $upload_result_signature = handle_upload('signature_file', $logged_in_principal_id, 'signature', $allowed_types, $max_file_size);

    // If any upload failed, stop and report the error
    if (!$upload_result_stamp['success'] || !$upload_result_signature['success']) {
        $status_message = "❌ Upload Failed: " . $upload_result_stamp['message'] . " " . $upload_result_signature['message'];
        $status_class = 'alert-error';
        goto skip_db_update;
    }

    // Determine the path to use: new upload path OR hidden field's existing path
    $final_stamp_path = $upload_result_stamp['path'] ?? $_POST['existing_stamp_path'];
    $final_signature_path = $upload_result_signature['path'] ?? $_POST['existing_signature_path'];

    // 4.3. Check if record exists
    $check_sql = "SELECT id FROM school_stamps WHERE principal_id = ?";
    $stmt_check = $conn->prepare($check_sql);
    $stmt_check->bind_param("s", $logged_in_principal_id); // 's' for string ID
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    
    if ($result_check->num_rows > 0) {
        // --- UPDATE Existing Record ---
        $update_sql = "
            UPDATE school_stamps 
            SET school_name = ?, stamp_image_path = ?, signature_image_path = ?
            WHERE principal_id = ?
        ";
        $stmt = $conn->prepare($update_sql);
        // 'ssss' for school_name, stamp_path, signature_path, principal_id
        $stmt->bind_param("ssss", $new_school_name, $final_stamp_path, $final_signature_path, $logged_in_principal_id);
        
        if ($stmt->execute()) {
            $status_message = "✅ Stamp and Signature details updated successfully!";
            $status_class = 'alert-success';
        } else {
            $status_message = "❌ Error updating details: " . $stmt->error;
            $status_class = 'alert-error';
        }
        $stmt->close();
        
    } else {
        // --- INSERT New Record ---
        $insert_sql = "
            INSERT INTO school_stamps 
            (school_name, stamp_image_path, signature_image_path, principal_id) 
            VALUES (?, ?, ?, ?)
        ";
        $stmt = $conn->prepare($insert_sql);
        // 'ssss' for school_name, stamp_path, signature_path, principal_id
        $stmt->bind_param("ssss", $new_school_name, $final_stamp_path, $final_signature_path, $logged_in_principal_id);
        
        if ($stmt->execute()) {
            $status_message = "✅ New Stamp and Signature details added successfully!";
            $status_class = 'alert-success';
        } else {
            $status_message = "❌ Error adding details: " . $stmt->error;
            $status_class = 'alert-error';
        }
        $stmt->close();
    }
    $stmt_check->close();

    // Re-assign form variables after submission attempt (to keep success/failure data visible)
    $school_name = $new_school_name;
    $stamp_path = $final_stamp_path;
    $signature_path = $final_signature_path;

    skip_db_update:
}


// ======================================================================
// 5. FETCH CURRENT DATA (for form pre-fill and display)
// ======================================================================

// Only attempt DB operations if connection is successful
if ($conn) {
    // Check if a record exists for the currently logged-in principal
    $fetch_sql = "
        SELECT school_name, stamp_image_path, signature_image_path 
        FROM school_stamps 
        WHERE principal_id = ?
    ";
    $stmt_fetch = $conn->prepare($fetch_sql);
    $stmt_fetch->bind_param("s", $logged_in_principal_id); // 's' for string ID
    $stmt_fetch->execute();
    $result_fetch = $stmt_fetch->get_result();

    if ($result_fetch->num_rows > 0) {
        $row = $result_fetch->fetch_assoc();
        // Use the fetched data unless it was overwritten by a successful POST
        if (!$status_message || $status_class == 'alert-error') {
            $school_name = htmlspecialchars($row['school_name']);
            $stamp_path = htmlspecialchars($row['stamp_image_path']);
            $signature_path = htmlspecialchars($row['signature_image_path']);
        } else {
            // If a successful post occurred, the variables already hold the new values
            $school_name = htmlspecialchars($school_name);
            $stamp_path = htmlspecialchars($stamp_path);
            $signature_path = htmlspecialchars($signature_path);
        }
        $record_exists = true;
    }

    $stmt_fetch->close();
    $conn->close(); // Close the database connection once fetching is complete
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Stamp and Signature Management</title>
    <style>
        /* Styling (identical to previous version) */
        :root {
            --primary-color: #007bff;
            --success-color: #28a745;
            --danger-color: #dc3545;
            --bg-light: #f8f9fa;
        }
        body { font-family: sans-serif; background-color: var(--bg-light); padding: 20px; }
        .container { max-width: 800px; margin: 0 auto; background: #fff; padding: 30px; border-radius: 12px; box-shadow: 0 10px 25px rgba(0,0,0,0.1); }
        h2 { color: var(--primary-color); border-bottom: 2px solid var(--primary-color); padding-bottom: 10px; margin-bottom: 30px; }
        
        .alert { padding: 15px; border-radius: 8px; margin-bottom: 20px; font-weight: 600; }
        .alert-error { color: #721c24; background-color: #f8d7da; border: 1px solid #f5c2c7; }
        .alert-success { color: #155724; background-color: #d4edda; border: 1px solid #c3e6cb; }

        .form-group { margin-bottom: 25px; }
        label { display: block; font-weight: 600; margin-bottom: 8px; color: #343a40; }
        input[type="text"] { width: 100%; padding: 12px; box-sizing: border-box; border: 1px solid #ced4da; border-radius: 8px; transition: border-color 0.3s; }
        
        /* File Input Styling */
        input[type="file"] { border: 1px solid #ced4da; padding: 10px; border-radius: 8px; width: 100%; box-sizing: border-box; background-color: #f5f5f5;}

        .btn-save { 
            background-color: var(--success-color); 
            color: white; 
            border: none; 
            cursor: pointer; 
            padding: 12px 25px; 
            border-radius: 8px; 
            font-weight: 700; 
            font-size: 17px; 
            transition: background-color 0.3s, transform 0.1s;
        }
        .btn-save:hover { background-color: #1e7e34; transform: translateY(-1px); }
        
        .image-preview-group { display: flex; gap: 20px; margin-top: 15px; }
        .image-preview-box { border: 1px solid #ddd; padding: 10px; border-radius: 8px; text-align: center; width: 50%; }
        .image-preview-box img { max-width: 100%; height: auto; max-height: 100px; object-fit: contain; margin-top: 10px; }
        .image-preview-box p { margin: 5px 0 0; font-size: 0.9em; color: #6c757d; }
    </style>
</head>
<body>

<div class="container">
    <h2>✍️ Manage School Stamp and Signature</h2>
    <p>Principal ID: **<?= htmlspecialchars($logged_in_principal_id) ?>** | Max File Size: **512KB** (JPG/PNG/GIF)</p>
    <hr>
    
    <?php if ($error): ?>
        <div class="alert alert-error">🚨 **Setup Error:** <?= htmlspecialchars($error) ?></div>
    <?php endif; ?>

    <?php if ($status_message): ?>
        <div class="alert <?= $status_class ?>"><?= htmlspecialchars($status_message) ?></div>
    <?php endif; ?>

    <h3><?= $record_exists ? 'Edit Existing Details' : 'Add New Details' ?></h3>
    
    <form method="POST" action="<?= htmlspecialchars($_SERVER['PHP_SELF']) ?>" target="_self" enctype="multipart/form-data">
        <input type="hidden" name="save_stamp_details" value="1">
        
        <div class="form-group">
            <label for="school_name">School Name (required):</label>
            <input type="text" id="school_name" name="school_name" value="<?= $school_name ?>" required placeholder="Enter Full School Name">
        </div>
        
        <div class="form-group">
            <label for="stamp_file">Upload Stamp Image (Optional to change):</label>
            <input type="file" id="stamp_file" name="stamp_file" accept="image/jpeg,image/png,image/gif">
            <input type="hidden" name="existing_stamp_path" value="<?= $stamp_path ?>">
        </div>
        
        <div class="form-group">
            <label for="signature_file">Upload Signature Image (Optional to change):</label>
            <input type="file" id="signature_file" name="signature_file" accept="image/jpeg,image/png,image/gif">
            <input type="hidden" name="existing_signature_path" value="<?= $signature_path ?>">
        </div>

        <div class="image-preview-group">
            <div class="image-preview-box">
                **Current Stamp**
                <?php if ($stamp_path && file_exists($stamp_path)): ?>
                    <img id="stamp_preview_img" src="<?= $stamp_path ?>" alt="Stamp Preview">
                <?php else: ?>
                    <p>No Stamp Image Saved</p>
                <?php endif; ?>
            </div>
            <div class="image-preview-box">
                **Current Signature**
                <?php if ($signature_path && file_exists($signature_path)): ?>
                    <img id="signature_preview_img" src="<?= $signature_path ?>" alt="Signature Preview">
                <?php else: ?>
                    <p>No Signature Image Saved</p>
                <?php endif; ?>
            </div>
        </div>
        
        <hr>
        <button type="submit" class="btn-save">
            <?= $record_exists ? '💾 Update Details' : '➕ Save New Details' ?>
        </button>
    </form>
</div>

<script>
// Live update image previews using FileReader on file selection
function setupPreview(fileInputId, imgElementId) {
    document.getElementById(fileInputId).addEventListener('change', function(event) {
        const file = event.target.files[0];
        const previewImg = document.getElementById(imgElementId);
        const box = previewImg ? previewImg.closest('.image-preview-box') : null;

        if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
                if (previewImg) {
                    previewImg.src = e.target.result;
                    previewImg.style.display = 'block'; 
                    const placeholder = box.querySelector('p');
                    if (placeholder) placeholder.style.display = 'none';
                }
            };
            reader.readAsDataURL(file);
        } else if (previewImg && box) {
            // If the user clears the file, reset the preview to the old saved image or placeholder
            const savedPath = document.querySelector(`input[name="existing_${fileInputId.replace('_file', '_path')}"]`).value;
            
            if (savedPath) {
                 previewImg.src = savedPath;
                 previewImg.style.display = 'block';
            } else {
                 // No saved path, hide image, show placeholder
                 previewImg.src = '';
                 previewImg.style.display = 'none';
                 const placeholder = box.querySelector('p');
                 if (placeholder) placeholder.style.display = 'block';
            }
        }
    });
}

// Initial setup
document.addEventListener('DOMContentLoaded', function() {
    setupPreview('stamp_file', 'stamp_preview_img');
    setupPreview('signature_file', 'signature_preview_img');
});
</script>

</body>
</html>